/* 
 * This file is part of the RTL examples distribution (https://tealsemi.com).
 * Copyright (c) 2022 Franz Steininger.
 * 
 * This program is free software: you can redistribute it and/or modify  
 * it under the terms of the GNU General Public License as published by  
 * the Free Software Foundation, version 3.
 *
 * This program is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

`default_nettype none

module tb;

reg [31:0] SIM_END_MASK;
reg [31:0] SIM_RPT_MASK;
integer my_seed;
initial begin
`ifdef POW2_SIM_END
  SIM_END_MASK = (1 << `POW2_SIM_END) - 1;
  SIM_END_MASK = (SIM_END_MASK | 32'h7f) & 32'h00ff_ffff;  // set min and max value
`else
  SIM_END_MASK = (1 << 18) - 1;
`endif
  SIM_RPT_MASK = SIM_END_MASK >> 6;
`ifdef SEED
  my_seed = `SEED;
`else
  my_seed = 1;
`endif
end

event       tick_start;
event       tick_end;
event       tick_display;
event       theEnd;

integer     stim_cnt;
integer     status;

reg         clk;

reg signed [15:0] a;
reg  [14:0] b;

wire        flag_must; 
wire        flag_ppa; 

initial begin
  stim_cnt = 0;
  status   = -1;
  clk      = 1'b0;
  a        = '0;
  b        = '0;
end

always begin
  #0 ->tick_start;
  #5 clk = !clk;
  #4 ->tick_end;
  #1 clk = !clk;
  stim_cnt = stim_cnt + 1;
end

test_must u_must(
  .a       ( a          ),  // input signed [15:0]
  .b       ( b          ),  // input  [15:0]
  .flag    ( flag_must  )   // output
);

test_ppa u_ppa (
  .a       ( a          ),  // input signed [15:0]
  .b       ( b          ),  // input  [15:0]
  .flag    ( flag_ppa   )   // output
);

// apply input vector
always @( tick_start ) begin
  if( stim_cnt < 64 ) begin
    casez( stim_cnt[5:0] )
      6'b000_000 : a = '0;
      6'b001_000 : a = '1;                // max positive signed or -1
      6'b010_000 : a = a >> 1;            // max positive (signed)
      6'b011_000 : a = ~a;                // max negative
      6'b1??_??? : a = $random(my_seed);  // random
    endcase
    case( stim_cnt[2:0] )
      3'b000 : b = '0;
      3'b001 : b = '1;                // max positive signed or -1
      3'b010 : b = b >> 1;            // max positive (signed)
      3'b011 : b = ~b;                // max negative
      default: b = $random(my_seed);  // random
    endcase
  end else begin
    a = $random(my_seed);
    b = $random(my_seed);
  end
end


// check outputs input vector
always @( tick_end ) begin
  if( flag_ppa !== flag_must ) begin
    status = 1;
    $display("Error:");
    ->tick_display;
    ->theEnd;
  end
  if( (stim_cnt & SIM_RPT_MASK) == SIM_RPT_MASK || stim_cnt <=100 ) begin
    ->tick_display;
  end
  if( (stim_cnt & SIM_END_MASK) == SIM_END_MASK ) begin
    ->theEnd;
  end
end


// check outputs input vector
always @( tick_display ) begin
  $display( "--->%9d: a=0x%4h;  b=0x%4h  --  flag_must=%b;  flag_ppa=%b", stim_cnt, a, b, flag_must,flag_ppa );
end


// finish simulation
always @( theEnd ) begin
  //$display( "---> SIM_END_MASK=0x%8h;  my_seed=0x%8h", SIM_END_MASK, my_seed );
  if( status == -1 ) begin
    status = 0;
    $display( "---> status: %1d  => Success !!! %1dk vectors (seed=%8h)\n", status, (stim_cnt+500)/1000, my_seed );
  end else begin
    $display( "---> status: %1d  => FAILED !!! %1dk vectors (seed=%8h)\n", status, (stim_cnt+500)/1000, my_seed );
  end
  $finish;
end

endmodule

`default_nettype wire

